#!/usr/bin/env python3
"""Plot results of the volume‑scaling sweep and extrapolate to infinite volume.

This module reads the CSV produced by
``scripts/run_volume_sweep_2ndpass.py``, groups the data by coupling
``b`` and exponent ``k`` and, for each group, fits the ratio
``R(L)`` versus ``1/L`` with a straight line.  The intercept of the
fit corresponds to the infinite–volume limit ``R(∞)``.  Individual
figures are saved for each ``(b, k)`` pair showing the data points
and the best–fit line.  A combined plot summarises all extrapolated
intercepts alongside the theoretical Casimir ratio ``8/3``.
"""

import os
import numpy as np
import pandas as pd
import matplotlib.pyplot as plt


def main() -> None:
    input_csv = os.path.join('results', 'volume_sweep_2ndpass.csv')
    if not os.path.exists(input_csv):
        raise FileNotFoundError(
            f"Results file '{input_csv}' not found. Please run the volume sweep first."
        )
    df = pd.read_csv(input_csv)

    # Ensure output directory exists
    os.makedirs('results', exist_ok=True)

    intercept_records = []
    # Iterate over all unique (b, k) pairs
    for (b, k), sub in df.groupby(['b', 'k']):
        # sort by L for consistent plotting
        sub_sorted = sub.sort_values('L')
        x = 1.0 / sub_sorted['L'].to_numpy()
        y = sub_sorted['R'].to_numpy()
        # Only perform a fit if at least two points exist
        if len(x) < 2:
            continue
        # Fit y = m x + c and extract intercept c
        # numpy.polyfit returns coefficients [slope, intercept]
        slope, intercept = np.polyfit(x, y, 1)
        intercept_records.append({'b': b, 'k': k, 'R_infty': intercept})

        # Create individual plot
        plt.figure()
        plt.scatter(x, y, color='tab:blue', label='data')
        # Evaluate fit line over a range of x including 0
        x_fit = np.linspace(0, x.max() * 1.1, 100)
        y_fit = slope * x_fit + intercept
        plt.plot(x_fit, y_fit, color='tab:orange', label=f'fit: R(∞)={intercept:.3f}')
        plt.xlabel('1/L')
        plt.ylabel('R(L)')
        plt.title(f'R vs 1/L for b={b}, k={k}')
        plt.legend()
        out_path = os.path.join('results', f'R_vs_invL_b{b}_k{k}.png')
        plt.savefig(out_path)
        plt.close()

    # Combined extrapolation plot
    if intercept_records:
        xs = range(len(intercept_records))
        ys = [rec['R_infty'] for rec in intercept_records]
        labels = [f"b={rec['b']}, k={rec['k']}" for rec in intercept_records]

        plt.figure(figsize=(max(6, len(xs) * 0.4), 4))
        plt.plot(xs, ys, 'o', markersize=6, label='Extrapolated R(∞)')
        # Theoretical line at 8/3
        theoretical = 8.0 / 3.0
        plt.axhline(theoretical, color='red', linestyle='--', label='Theoretical 8/3')
        plt.xticks(xs, labels, rotation=90)
        plt.ylabel('R(∞)')
        plt.title('Extrapolated infinite‑volume ratio across (b, k)')
        plt.legend()
        plt.tight_layout()
        combined_path = os.path.join('results', 'R_extrapolated_2ndpass.png')
        plt.savefig(combined_path)
        plt.close()
        print(f"Saved combined extrapolation plot to {combined_path}")
    else:
        print('No intercepts to plot.')


if __name__ == '__main__':
    main()